/* SPDX-License-Identifier: GPL-2.0 */
/*
 * Encoder core interface header
 *
 * Copyright (c) Imagination Technologies Ltd.
 * Copyright (c) 2021 Texas Instruments Incorporated - http://www.ti.com/
 *
 * Authors:
 *	Sunita Nadampalli <sunitan@ti.com>
 *
 * Re-written for upstreming
 *	Sidraya Jayagond <sidraya.bj@pathpartnertech.com>
 */

#ifndef __TOPAZ_API_H__
#define __TOPAZ_API_H__

#include <linux/types.h>
#include <linux/dma-mapping.h>
#include <media/v4l2-ctrls.h>
#include <media/v4l2-device.h>
#include <media/v4l2-mem2mem.h>

#include "fw_headers/topazscfwif.h"
#include "fw_headers/vxe_common.h"
#include "vid_buf.h"
#include "lst.h"

#define MAX_MVC_VIEWS		2
#define MVC_BASE_VIEW_IDX	0
#define NON_MVC_VIEW		(~0x0)

#define MVC_SPS_ID		1
#define MVC_PPS_ID		1

#define NUM_SLICE_TYPES		5
#define MAX_PLANES		3

/*
 * This type defines the buffer type categories.
 * @brief  Buffer Types
 */
enum venc_buf_type {
	VENC_BUFTYPE_BITSTREAM   = 0,
	VENC_BUFTYPE_PICTURE,
	VENC_BUFTYPE_ALL,
	VENC_BUFTYPE_MAX,
	VENC_BUFTYPE_FORCE32BITS = 0x7FFFFFFFU
};

/*
 * VXE callback type definitions
 */
enum vxe_cb_type {
	VXE_CB_CODED_BUFF_READY,
	VXE_CB_SRC_FRAME_RELEASE,
	VXE_CB_STR_END,
	VXE_CB_ERROR_FATAL,
	VXE_CB_FORCE32BITS = 0x7FFFFFFFU
};

typedef void (*vxe_cb)(void *ctx, enum vxe_cb_type type, void *buf_ref, unsigned int size,
		unsigned int coded_frm_cnt, enum img_frame_type frame_type);

/*
 * Enum specifying video encode profile
 */
enum img_video_enc_profile {
	ENC_PROFILE_DEFAULT     = 0,
	ENC_PROFILE_LOWCOMPLEXITY,
	ENC_PROFILE_HIGHCOMPLEXITY,
	ENC_PROFILE_REDUCEDMODE,
	ENC_PROFILE_FORCE32BITS = 0x7FFFFFFFU
};

/*
 * Enum describing smallest blocksize used during motion search
 */
enum img_ipe_minblock_size {
	BLK_SZ_16x16       = 0,
	BLK_SZ_8x8         = 1,
	BLK_SZ_4x4         = 2,
	BLK_SZ_DEFAULT     = 3,
	BLK_SZ_FORCE32BITS = 0x7FFFFFFFU
};

/*
 * Struct specifying flags to enable/disable encode features.
 * All boolean flags are FALSE by default
 */
struct img_encode_features {
	unsigned short disable_intra4x4;
	unsigned short disable_intra8x8;
	unsigned short disable_intra16x16;
	unsigned short disable_inter8x8;
	unsigned short restrict_inter4x4;
	unsigned short disable_bpic_ref1;
	unsigned short disable_bpic_ref0;
	unsigned short enable_8x16_mv_detect;
	unsigned short enable_16x8_mv_detect;
	unsigned short disable_bframes;
	enum img_ipe_minblock_size min_blk_sz;
	unsigned short restricted_intra_pred;
};

/*
 *
 * Struct describing Macro-block params generated by first stage.
 * Refer T.R.M. for details
 */
struct img_first_stage_mb_params {
	unsigned short ipe0_sad;
	unsigned short ipe1_sad;
	unsigned char ipe0_blks;
	unsigned char ipe1_blks;
	unsigned char carc_cmplx_val;
	unsigned char reserved;
};

/*
 * Size of Inter/Intra & Coded/Skipped tables
 */
#define TOPAZHP_SCALE_TBL_SZ                    (8)
#define DEFAULT_CABAC_DB_MARGIN    (0x190)

/*
 *Struct describing params for video encoding
 *@enable_sel_stats_flags:  Flags to enable selective first-pass statistics gathering by the
 *hardware. Bit 1 - First Stage Motion Search Data, Bit 2 - Best
 *			Multipass MB Decision Data, Bit 3 - Best Multipass Motion Vectors.
 *			(First stage Table 2 motion vectors are always switched on)
 *@enable_inp_ctrl:  Enable Macro-block input control
 *@enable_air:  Enable Adaptive Intra Refresh
 *@num_air_mbs:  n = Max number of AIR MBs per frame, 0 = _ALL_ MBs over threshold will be marked
 *			as AIR Intras, -1 = Auto 10%
 *@air_threshold:  n = SAD Threshold above which a MB is a AIR MB candidate,  -1 = Auto adjusting
 *			threshold
 *@air_skip_cnt:  n = Number of MBs to skip in AIR Table between frames, -1 = Random
 *			(0 - NumAIRMbs) skip between frames in AIR table
 *@disable_bit_stuffing:  Disabling bitstuffing to maintain bitrate
 *@mpeg2_intra_dc_precision:  Only used in MPEG2, 2 bit field (0 = 8 bit, 1 = 9 bit, 2 = 10 bit
 *			and 3=11 bit precision). Set to zero for other encode standards.
 *@enable_mvc: True if MVC is enabled. False by default
 *@mvc_view_idx:  MVC view index
 *@disable_bh_rounding:  True if we wish to disable the buffer height rounding to 16 pixels
 *			(enables contiguous YU memory for non-aligned image heights)
 *@auto_expand_pipes:  Automatically expand a context pipe allocations when new pipes become
 *			available
 *@line_counter_enabled:  <! Drives the activation of low-latency encoding
 */
struct img_video_params {
	enum img_standard standard;
	enum img_format format;
	enum img_csc_preset csc_preset;
	unsigned char slices_per_picture;
	unsigned short width;
	unsigned short frame_height;
	unsigned char is_interlaced;
	unsigned char is_interleaved;
	unsigned char constrained_intra;
	unsigned char h264_8x8;
	unsigned char bottom_field_first;
	unsigned char arbitrary_so;
	unsigned char cabac_enabled;
	unsigned int cabac_bin_limit;
	unsigned int cabac_bin_flex;
	unsigned char deblock_idc;
	unsigned char output_reconstructed;
	unsigned int f_code;
	int fine_y_search_size;
	unsigned char no_offscreen_mv;
	unsigned int idr_period;
	unsigned int intra_cnt;
	unsigned int vop_time_resolution;
	struct img_encode_features enc_features;
	unsigned char enable_sel_stats_flags;
	unsigned char enable_inp_ctrl;
	unsigned char enable_air;
	int num_air_mbs;
	int air_threshold;
	short air_skip_cnt;
	unsigned char enable_cumulative_biases;
	unsigned char enable_host_bias;
	unsigned char enable_host_qp;
	unsigned char use_default_scaling_list;
	short use_custom_scaling_lists;
	unsigned short pps_scaling;
	unsigned int vert_mv_limit;
	unsigned int intra_pred_modes;
	unsigned short limit_num_vectors;
	unsigned short disable_bit_stuffing;
	unsigned char coded_skipped_index;
	unsigned char inter_intra_index;
	unsigned char mpeg2_intra_dc_precision;

	/* Contents Adaptive Rate Control parameters */
	unsigned short carc;
	int carc_baseline;
	unsigned int carc_threshold;
	unsigned int carc_cutoff;
	unsigned int carc_neg_range;
	unsigned int carc_neg_scale;
	unsigned int carc_pos_range;
	unsigned int carc_pos_scale;
	unsigned int carc_shift;

	/* Weighted prediction */
	unsigned char weighted_prediction;
	unsigned char vp_weighted_implicit_bi_pred;

	/* SEI insertion */
	unsigned char insert_hrd_params;

	unsigned short intra_refresh;

	unsigned int chunks_per_mb;
	unsigned int max_chunks;
	unsigned int priority_chunks;
	unsigned int mbps;

	unsigned char multi_reference_p;
	unsigned char ref_spacing;
	unsigned char spatial_direct;

	unsigned short vp_adaptive_rounding_disable;
	unsigned short vp_adaptive_rounding_offsets[18][4];

	unsigned int debug_crcs;

	unsigned char enable_mvc;
	unsigned short mvc_view_idx;
	unsigned char high_latency;

	unsigned short buffer_stride_bytes;
	unsigned short buffer_height;
	unsigned char disable_bh_rounding;
	unsigned short source_width;
	unsigned short source_frame_height;

	unsigned char no_sequence_headers;
	unsigned char auto_encode;
	unsigned char slice_level;
	unsigned char coded_header_per_slice;

	unsigned char auto_expand_pipes;
	unsigned char enable_lossless;
	unsigned char lossless_8x8_prefilter;

	unsigned char enable_scaler;
	unsigned short crop_left;
	unsigned short crop_right;
	unsigned short crop_top;
	unsigned short crop_bottom;

#if SECURE_IO_PORTS
	unsigned char secure_ctx_input;
	unsigned char secure_ctx_output;
#endif
	/* Low latency encoding related */
	unsigned char line_counter_enabled;
};

/*
 * Struct describing JPEG component info
 */
struct component_info {
	unsigned int width;           /* Width of the image component */
	unsigned int stride;          /* Stride of the image component */
	unsigned int step;            /* Step of the image component */
	unsigned int height;          /* Height of the image component */
};

/*
 * Struct describing JPEG info
 */
struct img_unaligned_source_info {
	unsigned int output_width;                /* Width of the JPEG image */
	unsigned int output_height;               /* Height of the JPEG image */
	unsigned int components;                  /* Number of components in the image ( 1 or 3 ) */
	struct component_info comp_info[3]; /* Array containing component info */
};

/*
 * Struct containing details of a reconstuctured picture
 */
struct img_recon_done {
	unsigned int poc;		/* PicOrderCount */
	void *buffer;		/* Buffer containing reconstructured image */
};

/*
 * Struct containing a feedback for one frame
 *@is_coded:  Is the frame was coded or skipped completely?
 *@is_skipped:  Is the frame coded as a set of skipped MBs?
 *@entire_frame:  Was the frame encoded entirely or there a still slices to come?
 *@img_frame_type frame_type:  Frame Type (IDR, I, P, B)
 *@source_slot:  Number of the source slot, containing the source buffer
 *@recon_idx:  Number of the slot, conaining reconstructed picture
 *@*src_frame:  Source buffer
 *@*motion_search_statistics_buf:  Buffer to contain Table 1 (8 byte Motion Search Data) of the
 *					selectable stats
 *@*best_multipass_statistics_buf:  Buffer to contain (optionally) Table 4 (64 bytes
 *				Multipass Motion Vectors) and table 3 (16 byte Multipass Parameters)
 *@coded_package: Pointer to the coded package, containing the returned
 *			header and coded buffer information
 *@recon_list:  List of reconstructed pictures
 *@bytes_coded:  Size of the encoded slice in bytes
 *@first_bu:  Number of the first BU in the slice
 *@storage_frame_num:  Last 2 bits of the Frame number in Storage Order
 *@slice_num:  Number of the slice in a Slice Map supplied
 *@slices_per_picture:  Number of the slices in this picture
 *@field:  The field this slice belongs to
 *@coded_slot_num:  Slot number of the coded buffer
 *@poc:  PicOrderCount of the coded slice
 *@patched_recon:  Was the reconstructed picture written to a patched buffer?
 *@slices_in_buffer:  Number of slices contained in the coded buffer
 *@last_frame_encoded:  True if the last frame has been encoded
 *@coded_buffer_count:  Number of coded buffer used
 *@host_ctx:  Host context value
 */
struct img_feedback_element {
	unsigned char is_coded;
	unsigned char is_skipped;
	unsigned char entire_frame;
	enum img_frame_type frame_type;
	unsigned char source_slot;
	unsigned char recon_idx;
	struct img_frame *src_frame;
	struct img_buffer *motion_search_statistics_buf;
	struct img_buffer *best_multipass_statistics_buf;

	struct coded_package_host *coded_package;

	struct list_item *recon_list;
	unsigned int bytes_coded;
	unsigned int first_bu;
	unsigned char storage_frame_num;
	unsigned char slice_num;
	unsigned char slices_per_picture;
	unsigned char field;
	unsigned char coded_slot_num;

	unsigned char active_coded_package_idx;

	unsigned int poc;
	unsigned char patched_recon;
	unsigned char slices_in_buffer;
	unsigned char last_frame_encoded;
	unsigned char coded_buffer_count;

	unsigned long long host_ctx;
};

/*
 * Bit fields for ui32CoreFeatures
 */
#define SCALER_SUPPORTED_MASK                           0x00000080
#define GENERATE_PERFORMANCE_STORE_MASK                 0x00000100
#define H264_LOSSLESS_SUPPORTED_MASK                    0x00000200
#define H264_CUSTOM_QUANT_SUPPORTED_MASK                0x00000400
#define MPEG2_SUPPORTED_MASK                            0x00000800
#define SIGNATURES_SUPPORTED_SUBSET_MASK                0x00001000
#define SIGNATURES_SUPPORTED_ALL_MASK                   0x00002000
#define H264_WEIGHTED_PRED_ME_SUPPORTED_MASK		0x00004000
#define H264_WEIGHTED_PRED_SUPPORTED_MASK               0x00008000
#define H264_2_REF_ON_P_PIC_SUPPORTED_MASK              0x00010000
#define H264_SPATIAL_DIRECT_SUPPORTED_MASK              0x00020000
#define H264_MULTIPASS_SUPPORTED_MASK                   0x00040000
#define H264_DEFAULT_TABLES_SUPPORTED_MASK              0x00080000
#define H264_8X8_TRANSFORM_SUPPORTED_MASK               0x00100000
#define H264_INTERLACED_SUPPORTED_MASK         0x00200000
#define H264_B_PIC_SUPPORTED_MASK              0x00400000
#define H264_16X8_8X16_SUPPORTED_MASK          0x00800000
#define H264_CABAC_SUPPORTED_MASK              0x01000000
#define SLAVE_JPEG_SUPPORTED_MASK              0x02000000
#define JPEG_SUPPORTED_MASK           0x04000000
#define H263_SUPPORTED_MASK           0x08000000
#define MPEG4_SUPPORTED_MASK          0x10000000
#define H264_SUPPORTED_MASK           0x20000000
#define DMAC_SUPPORTED_MASK           0x40000000
#define MMU_SUPPORTED_MASK            0x80000000

/*
 * Struct describing the capabilities of the encoder
 */
struct img_enc_caps {
	unsigned short min_slices;              /* Minimum slices to use */
	unsigned short max_slices;              /* Maximum slices to use */
	unsigned short recommended_slices;      /* Recommended number of slices */
	unsigned short num_cores;               /* Number of cores that will be used */
	unsigned int core_features;           /* Core features flags */
	unsigned int core_revision;           /* Core revision */
	unsigned int max_height;              /* Maximum height supported */
	unsigned int max_width;               /* Maximum width supported */
	unsigned int min_height;              /* Minimum height supported */
	unsigned int min_width;               /* Minimum width supported */
	unsigned int max_mb_num;              /* Maximum number of macro blocks */
	unsigned int min_slice_height;        /* Minimum number of rows that a slice can span
					       * (not including automatic slice breaks)
					       */
	unsigned int max_bu_per_frame;        /* Maximum number of basic units per frame */
};

/*
 * Struct describing driver results message
 */
struct driver_tohost_msg {
	enum mtx_message_id cmd_id;
	enum mtx_cmd_id input_cmd_id;
	unsigned int data;
	struct vidio_ddbufinfo *command_data_buf;
	struct img_feedback_element feedback;
};

/*
 * Enum describing picture coding type
 */

enum h264_picture_coding_type {
	I_FRAME                = 0, /* An intra frame */
	P_FRAME                = 1, /* An inter frame */
	B_FRAME                = 2, /* A B frame */
	FRAME_TYPE_FORCE32BITS = 0x7FFFFFFFU
};

/*
 * Struct describing H264 VUI parameters
 */
struct h264_vui_params {
	unsigned int time_scale;	/* Time scale as defined in the H.264 specification */
	unsigned int bit_rate_value_minus1;		/* An inter framebitrate/64)-1 */
	unsigned int cbp_size_value_minus1;		/* An inter frame(bitrate*1.5)/16 */
	unsigned char aspect_ratio_info_present_flag;	/* aspect_ratio_info_present_flag as
							 * defined in the H.264 specification
							 */
	unsigned char aspect_ratio_idc;			/* as defined in the H.264 specification */
	unsigned short sar_width;
	unsigned short sar_height;
	unsigned char cbr;			/* CBR as defined in the H.264 specification */
	/* as defined in the H.264 specification */
	unsigned char initial_cpb_removal_delay_length_minus1;
	unsigned char cpb_removal_delay_length_minus1;	/* as defined in the H.264 specification */
	unsigned char dpb_output_delay_length_minus1;	/* as defined in the H.264 specification */
	unsigned char time_offset_length;		/* as defined in the H.264 specification */
	unsigned char num_reorder_frames;
	unsigned char max_dec_frame_buffering;
};

/*
 * Struct describing H264 crop parameters
 */
struct h264_crop_params {
	/* Flag indicating if cropping parameters are present */
	unsigned char clip;
	/* Number of pixels to crop from the left side /2 */
	unsigned short left_crop_offset;
	/* Number of pixels to crop from the right side /2 */
	unsigned short right_crop_offset;
	unsigned short top_crop_offset;		/* Number of pixels to crop from the top /2 */
	unsigned short bottom_crop_offset;	/* Number of pixels to crop from the bottom /2 */
};

/*
 * Enum describing Profile (H264)
 */
enum sh_profile_type {
	SH_PROFILE_BP          = 0,	/* H.264 Baseline Profile */
	SH_PROFILE_MP          = 1,	/* H.264 Main Profile */
	SH_PROFILE_HP          = 2,	/* H.264 High Profile */
	SH_PROFILE_H10P        = 3,	/* H.264 High 10P Profile */
	SH_PROFILE_H422P       = 4,	/* H.264 High 4:2:2 Profile */
	SH_PROFILE_H444P       = 5,	/* H.264 High 4:4:4 Profile */
	SH_PROFILE_FORCE32BITS = 0x7FFFFFFFU
};

/*
 * Enum describing level (H264)
 */
enum sh_level_type {
	SH_LEVEL_1           = 10,	/* H264 Level as specified in the specification */
	SH_LEVEL_1B          = 1,	/* Special-case H264 Level */
	SH_LEVEL_11          = 11,	/* H264 Level as specified in the specification */
	SH_LEVEL_12          = 12,	/* H264 Level as specified in the specification */
	SH_LEVEL_13          = 13,	/* H264 Level as specified in the specification */
	SH_LEVEL_2           = 20,	/* H264 Level as specified in the specification */
	SH_LEVEL_21          = 21,	/* H264 Level as specified in the specification */
	SH_LEVEL_22          = 22,	/* H264 Level as specified in the specification */
	SH_LEVEL_3           = 30,	/* H264 Level as specified in the specification */
	SH_LEVEL_31          = 31,	/* H264 Level as specified in the specification */
	SH_LEVEL_32          = 32,	/* H264 Level as specified in the specification */
	SH_LEVEL_4           = 40,	/* H264 Level as specified in the specification */
	SH_LEVEL_41          = 41,	/* H264 Level as specified in the specification */
	SH_LEVEL_42          = 42,	/* H264 Level as specified in the specification */
	SH_LEVEL_5           = 50,	/* H264 Level as specified in the specification */
	SH_LEVEL_51          = 51,	/* H264 Level as specified in the specification */
	SH_LEVEL_52          = 52,	/* H264 Level as specified in the specification */
	SH_LEVEL_FORCE32BITS = 0x7FFFFFFFU
};

/*
 * Struct describing SPS (sequence) parameters (H264)
 */
struct h264_sequence_header_params {
	enum sh_profile_type profile;
	enum sh_level_type level;
	unsigned char width_in_mbs_minus1;
	unsigned char height_in_maps_units_minus1;
	unsigned char log2_max_pic_order_cnt;
	unsigned char max_num_ref_frames;
	unsigned char gaps_in_frame_num_value;
	unsigned char frame_mbs_only_flag;
	unsigned char vui_params_present;
	unsigned char seq_scaling_matrix_present_flag;

	unsigned char use_default_scaling_list;
	unsigned char is_lossless;
	struct h264_vui_params vui_params;
};

/*
 * Struct describing Bias parameters
 */
struct img_henc_debug_settings {
	unsigned int serialized_communication_mode;
};

#define VXE_SERIALIZED_MODE_OFF         (0)
#define VXE_SERIALIZED_MODE_SYNCED      (1)
#define VXE_SERIALIZED_MODE_SERIAL      (2)

/*
 * Struct describing input parameters to encode a video picture
 */
struct pic_params {
	unsigned int flags;                           /* Picture parameter flags */
	struct in_rc_params in_params;          /* Rate control parameters */
};

/*
 * Video encode context
 *@void *dd_str_ctx:		Pointer to device instance data
 *@standard:			Video standard
 *@frame_height:		target output height
 *@picture_height:		target output height
 *@buffer_stride_bytes:		input buffer stride
 *@buffer_height:		input buffer width
 *@format:	Pixel format of the source surface
 *@csc_preset:	Colour space conversion to be performed on the source surface
 *@pic_params:	Picture level parameters (supplied by driver)
 *@above_params[TOPAZHP_MAX_NUM_PIPES]:	Picture level parameters (supplied by driver)
 *@recon_pictures[MAX_PIC_NODES]:  Reference pictures (2 input and 1 output)
 *@colocated[MAX_PIC_NODES]:	Colocated vector stores (2 input and 1 output)
 *@mv[MAX_MV:	Vector stores
 *@inter_view_mv[2]:  Inter-view vector stores
 *@seq_header_mem:  Partially coded Sequence header
 *@subset_seq_header_mem:  Partially coded Subset sequence header for H264 mvc
 *@flat_gop_struct:  Flat MiniGop structure
 *@hierar_gop_struct:  Hierarchical MiniGop structure
 *@custom_quant[2]:  Custom quantization values
 *@custom_quant_regs4x4_sp[2]:	Custom quantization register values for 4x4 Sp
 *@vidio_ddbufinfo custom_quant_regs8x8_sp[2]:	Custom quantization register values for 8x8 Sp
 *@vidio_ddbufinfo custom_quant_regs4x4_q[2]:	Custom quantization register values for 4x4 Q
 *@vidio_ddbufinfo custom_quant_regs8x8_q[2]:	Custom quantization register values for 8x8 Q
 *@slice_map[MAX_SOURCE_SLOTS_SL]:	Slice map of the source picture
 *@firstpass_out_param_buf[MAX_SOURCE_SLOTS_SL]:	Output Parameters of the First Pass
 *@firstpass_out_best_multipass_param_buf[MAX_SOURCE_SLOTS_SL]:	Output Selectable Best MV
 *								Parameters of the First Pass
 *@mb_ctrl_in_params_buf[MAX_SOURCE_SLOTS_SL]:	Input Parameters to the second pass
 *@ipe_control:	common bits IPE control register for entire picture
 *@pred_comb_control:  common bits of Predictor-combiner control register for entire picture
 *@cabac_enabled:  FLAG to enable Cabac mode
 *@cabac_bin_limit:  Min Bin Limit after which the Topaz hardware would encode MB as IPCM
 *@cabac_bin_flex:  Max Flex-Limit, the Topaz-HW will encode MB as IPCM after (BinLimit+BinFlex)
 *@vidio_ddbufinfo mv_settings_btable:	three colocated vector stores (2 input and 1 output)
 *@img_frame *source_slot_buff[MAX_SOURCE_SLOTS_SL]:	Source slots
 *@unsigned int source_slot_poc[MAX_SOURCE_SLOTS_SL]:	POCs of frames in slots
 *@unsigned char slots_in_use:	Number of source slots
 *@unsigned char slots_required:  Number of source slots to be consumed
 *@coded_package_host *coded_package[MAX_CODED_PACKAGES]:  Collection of coded/header information
 *@unsigned char encoder_idle:  Indicates that the encoder is waiting for data,
 *                              Set to true at start of encode
 *@unsigned char enable_sel_stats_flags:  Flags to enable selective first-pass
 *					statistics gathering by the
 *					hardware. Bit 1 - First Stage Motion Search Data, Bit 2
 *					- Best Multipass MB Decision Data, Bit 3 - Best Multipass
 *					 Motion Vectors. (First stage Table 2 motion vectors are
 *					 always switched on)
 *@enable_inp_ctrl;Enable Macro-block input control
 *@enable_air:	Enable Adaptive Intra Refresh
 *@num_air_mbs:	n = Max number of AIR MBs per frame, 0 = _ALL_ MBs over threshold will be marked
 *				as AIR Intras, -1 = Auto 10%
 *@air_threshold:  n = SAD Threshold above which a MB is a AIR MB candidate,
 *				-1 = Auto adjusting threshold
 *@air_skip_cnt:  n = Number of MBs to skip in AIR Table between frames,
 *			-1 = Random (0 - NumAIRMbs) skip between frames in AIR table
 *@enable_mvc:		True if MVC is enabled. False by default
 *@mvc_view_idx:	View Idx of this MVC view
 *@line_counter:	Keep track of line counter activation
 */
struct img_video_context {
	/* topaz dd str context handle */
	void *dd_str_ctx;
	unsigned int dd_ctx_num;

	/* stream level params */
	enum img_standard standard;
	unsigned short width;
	unsigned short frame_height;
	unsigned short picture_height;
	unsigned short buffer_stride_bytes;
	unsigned short buffer_height;
	unsigned char frame_rate;

	unsigned short unrounded_width;
	unsigned short unrounded_frame_height;

	unsigned int debug_crcs;
	enum img_format format;
	enum img_csc_preset csc_preset;

	/* Numbers of array elements that will be allocated */
	int pic_nodes;
	int mv_stores;

	/* per core params */
	struct pic_params pic_params;
	struct vidio_ddbufinfo above_params[TOPAZHP_MAX_NUM_PIPES];
	struct vidio_ddbufinfo recon_pictures[MAX_PIC_NODES];
	struct vidio_ddbufinfo colocated[MAX_PIC_NODES];
	struct vidio_ddbufinfo mv[MAX_MV];
	struct vidio_ddbufinfo inter_view_mv[2];

	/* partially coded headers supplied to HW */
	/* SEI_INSERTION */
	struct vidio_ddbufinfo aud_header_mem;
	struct vidio_ddbufinfo sei_buffering_period_header_mem;
	struct vidio_ddbufinfo sei_picture_timing_header_mem;

	struct vidio_ddbufinfo seq_header_mem;
	struct vidio_ddbufinfo subset_seq_header_mem;
	struct vidio_ddbufinfo pichdr_template_mem[4];
	struct vidio_ddbufinfo slice_params_template_mem[NUM_SLICE_TYPES];

	unsigned int f_code;
	struct vidio_ddbufinfo src_phys_addr;

	/* WEIGHTED PREDICTION */
	struct vidio_ddbufinfo weighted_prediction_mem[MAX_SOURCE_SLOTS_SL];
	unsigned char weighted_prediction;
	unsigned char weighted_bi_pred;

	struct vidio_ddbufinfo flat_gop_struct;
	struct vidio_ddbufinfo hierar_gop_struct;

	struct vidio_ddbufinfo ltref_header[MAX_SOURCE_SLOTS_SL];

	struct vidio_ddbufinfo custom_quant[2];
	struct vidio_ddbufinfo custom_quant_regs4x4_sp[2];
	struct vidio_ddbufinfo custom_quant_regs8x8_sp[2];
	struct vidio_ddbufinfo custom_quant_regs4x4_q[2];
	struct vidio_ddbufinfo custom_quant_regs8x8_q[2];
	unsigned char custom_quant_slot;

	struct img_buffer slice_map[MAX_SOURCE_SLOTS_SL];

	struct img_buffer firstpass_out_param_buf[MAX_SOURCE_SLOTS_SL];

	struct img_buffer firstpass_out_best_multipass_param_buf[MAX_SOURCE_SLOTS_SL];
	struct img_buffer mb_ctrl_in_params_buf[MAX_SOURCE_SLOTS_SL];

	/* these values set at picture level & written in at slice */
	unsigned int ipe_control;
	unsigned int pred_comb_control;
	unsigned char cabac_enabled;
	unsigned int cabac_bin_limit;
	unsigned int cabac_bin_flex;

	unsigned int first_pic_flags;
	unsigned int non_first_pic_flags;

	unsigned char is_interlaced;
	unsigned char is_interleaved;
	unsigned char top_field_first;
	unsigned char arbitrary_so;
	unsigned char slices_per_picture;
	unsigned char deblock_idc;
	unsigned int kick_size;
	unsigned int kicks_per_bu;
	unsigned int vop_time_resolution;
	unsigned int idr_period;
	unsigned int intra_cnt;
	unsigned char multi_reference_p;
	unsigned char spatial_direct;

	struct img_mv_settings mv_settings_idr;
	struct img_mv_settings mv_settings_non_b[MAX_BFRAMES + 1];

	/*  | MVSetingsB0 | MVSetingsB1 | ... | MVSetings Bn |  */

	struct vidio_ddbufinfo mv_settings_btable;
	struct vidio_ddbufinfo mv_settings_hierarchical;

	/* Source slots */
	struct img_frame *source_slot_buff[MAX_SOURCE_SLOTS_SL];
	unsigned int source_slot_poc[MAX_SOURCE_SLOTS_SL];
	unsigned char slots_in_use;
	unsigned char slots_required;

	/* Coded slots */
	struct coded_package_host *coded_package[MAX_CODED_PACKAGES];
	unsigned int coded_buffer_max_size;
	unsigned char coded_package_max_num;

	unsigned int frame_count;
	unsigned int flush_at_frame;
	unsigned int flushed_at_frame;
	unsigned int encode_sent;
	unsigned int encode_requested;
	unsigned int frames_encoded;
	unsigned char encoder_idle;
	unsigned char aborted;

	struct list_item *ref_frame;
	unsigned int recon_poc;
	unsigned int next_recon;

	struct vidio_ddbufinfo *recon_buffer;
	struct vidio_ddbufinfo *patched_recon_buffer;

	struct img_rc_params rc_params;
	enum img_frame_type frame_type;

	unsigned int buffers_status_reg;

	unsigned char insert_seq_header;
	unsigned char output_reconstructed;

	unsigned int encode_pic_processing;
	unsigned char extra_wb_retrieved;

	unsigned char enable_sel_stats_flags;

	unsigned char enable_inp_ctrl;
	unsigned char enable_air;
	int num_air_mbs;
	int air_threshold;
	short air_skip_cnt;

	unsigned char enable_host_bias;
	unsigned char enable_host_qp;

	unsigned char custom_scaling;
	unsigned char pps_scaling;
	unsigned char h264_8x8_transform;
	unsigned char h264_intra_constrained;
	unsigned int vert_mv_limit;
	unsigned int intra_pred_modes;
	unsigned char limit_num_vectors;
	unsigned char disable_bit_stuffing;
	unsigned char coded_skipped_index;
	unsigned char inter_intra_index;
	struct vidio_ddbufinfo mtx_enc_ctx_mem;
	/* SEI_INSERTION */
	unsigned char insert_hrd_params;
	unsigned int chunks_per_mb;
	unsigned int max_chunks;
	unsigned int priority_chunks;

	unsigned char source_slot_reserved;
	unsigned char coded_package_slot_reserved;
	void *encode_pic_signal;

	unsigned char highest_storage_number;
	unsigned char vid_ctx_num;
	unsigned char enable_mvc;
	unsigned short mvc_view_idx;
	unsigned char high_latency;
	unsigned int mbps;

	unsigned char no_sequence_headers;
	unsigned int next_slice;
	unsigned char auto_encode;
	unsigned char slice_level;
	unsigned char coded_header_per_slice;

	/* Scaler specific values */
	unsigned char enable_scaler;
	unsigned short crop_left;
	unsigned short crop_right;
	unsigned short crop_top;
	unsigned short crop_bottom;
	unsigned short source_width;
	unsigned short source_frame_height;

#if SECURE_IO_PORTS
	unsigned char secure_ctx_input;
	unsigned char secure_ctx_output;
#endif
	unsigned char line_counter;
};

/*
 * Encoder context
 *@codec:  encode codec
 *@video:  Video encode context
 *@base_pip:  The first of a contiguous set of pipes to use for the context encode
 *@pipes_to_use:  The number of contiguous pipes (starting with ui8BasePipe) to
 *			use for the context encode
 *@requested_base_pipe:  The first of a contiguous set of pipes to use for the context encode
 *@requested_pipes_to_use:  The number of contiguous pipes (starting with ui8BasePipe)
 *				to use for the context encode
 *@auto_expand_pipes:  Automatically expand a context pipe allocations
 *			when new pipes become available
 *@sync_first_pass:  true if never synced
 */
struct img_enc_context {
	enum img_codec codec;
	struct img_video_context *video;

	unsigned char base_pipe;
	unsigned char pipes_to_use;

	unsigned char requested_base_pipe;
	unsigned char requested_pipes_to_use;
	unsigned char auto_expand_pipes;

	unsigned char ctx_num;

	unsigned char sync_first_pass;
	unsigned int core_rev;

	struct img_enc_caps caps;
	struct img_henc_debug_settings *debug_settings;
};

/*
 * Struct containing details of a reconstuctured picture
 */
struct img_recon_node {
	unsigned int poc;             /* PicOrderCount */
	void *buffer;           /* Buffer containing reconstructured image */
};

/*
 * This structure contains the topaz Context.
 * @brief  topaz Context
 */
struct topaz_core_context {
	/* List of stream context structures */
	struct topaz_dev_ctx *dev_handle;
	struct lst_t topaz_stream_list;
	vxe_cb vxe_str_processed_cb;
	unsigned int num_pipes;
	struct mutex *mutex;
};

struct topaz_stream_context {
	void **link;            /* List link (allows the structure to be part of a MeOS list).*/
	struct topaz_core_context *core_ctx;
	struct img_enc_context *enc_ctx;
	unsigned int stream_id; /* DMAN allocated device ID. */
	struct vxe_enc_ctx *vxe_ctx;
	void *mmu_ctx;    /* stream specific MMU context */

};

/*
 * Function pointer type for picture management functions
 */
typedef void (*pic_mgmt_func)(void *app_context, unsigned int frame_num);

/*
 * @function	InitHardware
 * @brief	Initialise the Encoder Hardware
 * @details	Reset the hardware and set up registers, etc.
 */
int init_topaz_core(void *dev_handle, unsigned int *num_pipes,
		    unsigned int mmu_flags, void *callback);

/*
 * @function		DeinitHardware
 */
int  deinit_topaz_core(void);

/*
 * @function		CreateContext
 * @brief		Create an encoder context
 * @details		Set up an encoder context with the given parameters
 */
int topaz_stream_create(void *vxe_ctx, struct img_video_params *video_params,
			unsigned char base_pipe,
			unsigned char pipes_to_use, struct img_rc_params *rc_params,
			void **topaz_str_context);

int topaz_end_of_stream(void *topaz_str_ctx, unsigned int frame_cnt);

int topaz_flush_stream(void *topaz_str_ctx, unsigned int frame_cnt);

int topaz_stream_destroy(void *topaz_str_ctx);

/*
 * Load the given context onto the hardware
 */
int topaz_load_context(void *topaz_str_ctx);

/*
 * Store the context from the hardware into given location
 */
int topaz_store_context(void *topaz_str_ctx);

/*
 * Destroy the given context onto the hardware
 */
int topaz_destroy_context(void *topaz_str_ctx);

/*
 * Get the capabilities of the encoder for the given requirements.
 * @param    standard		: Standard setting
 * @param    width		: Target output width
 * @param    height		: Target output height
 * @param    caps		: Pointer to caps structure to be filled in.
 */
int topaz_get_encoder_caps(enum img_standard standard, unsigned short width,
			   unsigned short height, struct img_enc_caps *caps);

int topaz_stream_map_buf_sg(void *topaz_str_ctx, enum venc_buf_type buf_type,
			    struct vidio_ddbufinfo *buf_info, void *sgt);

int topaz_stream_unmap_buf_sg(void *topaz_str_ctx, struct vidio_ddbufinfo *buf_info);

/*
 * Prepare a partially coded H264 Sequence Header (SPS).
 * @param    mb_width				: Width of the sequence in MBs
 * @param    mb_height			: Height of the sequence in MBs
 * @param    vui_params_present	: IMG_TRUE to include VUI parameters
 * @param    params				: Pointer to VUI parameters structure
 * @param    crop				: Pointer to crop parameter structure
 * @param    sh_params			: Pointer to sequence header params structure
 */
int topaz_h264_prepare_sequence_header(void *topaz_str_ctx, unsigned int mb_width,
				       unsigned int mb_height,
				       unsigned char vui_params_present,
				       struct h264_vui_params *params,
				       struct h264_crop_params *crop,
				       struct h264_sequence_header_params *sh_params,
				       unsigned char mvc_sps);
/*
 * Prepare a partially coded H264 Picture Header (PPS).
 * @param		cqp_offset			: Chroma QP offset
 */
int topaz_h264_prepare_picture_header(void *topaz_str_ctx, signed char cqp_offset);

/*
 * Prepare an AUD header.
 */
int topaz_h264_prepare_aud_header(void *topaz_str_ctx);

/*
 * Set offsets and strides for YUV components.
 * @param		frame				: Source frame
 */
int topaz_set_component_offsets(void *topaz_str_ctx, struct img_frame *frame);

/*
 * Reserves a slot to be used by a subsequent call to SendCodedPackageToFW.
 * If internally allocated coded buffers are being used, it also gets a buffer
 * from the encoder's internal list.
 */
int topaz_reserve_coded_package_slot(void *topaz_str_ctx);

/*
 * Submits a buffer to the encoder to receive coded data along with a coded header buffer.
 * A VP8 non coded buffer can also be sent.
 * @param	coded_buffer			: Pointer to the coded package to send to FW
 */
int topaz_send_coded_package(void *topaz_str_ctx, struct img_coded_buffer *coded_buffer);

/*
 * Returns the number of empty source slots available
 * @return   unsigned char: Number of empty source slots  (negative number indicates an error)
 */
signed char topaz_query_empty_source_slots(void *topaz_str_ctx);

/*
 * Returns the number of empty coded buffer slots available
 * @return   signed char: Number of empty coded slots (negative number indicates an error)
 */
signed char topaz_query_empty_coded_slots(void *topaz_str_ctx);

/*
 * Reserves a slot to be used by a subsequent call to SendSourceFrame.
 * If internally allocated source buffers are being used, it also gets a buffer
 * from the encoder's internal list.
 * @param		src_slot_num		: Pointer to receive slot number
 */
int topaz_reserve_source_slot(void *topaz_str_ctx, unsigned char *src_slot_num);

/*
 * Submits a frame to the encoder for processing
 * @param		src_frame			: Pointer to receive buffer pointer
 * @param		frame_num		: Frame number of the given frame
 * @param		ctx_value	: Value which will be insert into coded data buffer header
 */
int topaz_send_source_frame(void *topaz_str_ctx,
			    struct img_frame *src_frame,
			    unsigned int frame_num,
			    unsigned long long ctx_value);

/*
 * Indicates that there are no more source frames to send. It may not be the last
 * command, but it should inform the drivers about the length of the video stream.
 */
int topaz_end_of_stream(void *topaz_str_ctx, unsigned int frame_count);

/*
 * Indicates that the encoder should be flushed after the specified number of
 * frames have been encoded.
 * @param    frame_count: Number of frames which should have been encoded
 *		 when the flush is complete
 */
int topaz_flush_stream(void *topaz_str_ctx, unsigned int frame_count);

/*
 * Get the maximum coded data length for the given parameters, which can be used
 * to determine the size of the coded data buffer.
 */
unsigned int topaz_get_coded_buffer_max_size(void *topaz_str_ctx, enum img_standard standard,
					     unsigned short width, unsigned short height,
					     struct img_rc_params *rc_params);

unsigned int topaz_get_coded_package_max_num(void *topaz_str_ctx, enum img_standard standard,
					     unsigned short width, unsigned short height,
					     struct img_rc_params *rc_params);

/*
 * Tell the firmware to encode a frame.
 */
int topaz_encode_frame(void *topaz_str_ctx);

/*
 * Tells whether or not a pipe is being used by any context.
 * If it is being used then it returns the id (1 or 2) of the context that is using it.
 * Else it returns zero as the context id.
 */
int topaz_get_pipe_usage(unsigned char pipe, unsigned char *ctx_id);

#endif
